/*
 * Copyright (C) Peel Technologies Inc
 */

#ifndef PEELIR_H
#define PEELIR_H

#include <linux/types.h>

/* User space versions of kernel symbols for SPI clocking modes,
 * matching <linux/spi/spi.h>
 */

#define SPI_CPHA		0x01
#define SPI_CPOL		0x02

#define SPI_CS_HIGH		0x04
#define SPI_LSB_FIRST		0x08
#define SPI_3WIRE		0x10
#define SPI_LOOP		0x20
#define SPI_NO_CS		0x40
#define SPI_READY		0x80

/*---------------------------------------------------------------------------*/

/* IOCTL commands */

#define SPI_IOC_MAGIC			'k'

/**
 * struct spi_ioc_transfer - describes a single SPI transfer
 * @tx_buf: Holds pointer to userspace buffer with transmit data, or null.
 *	If no data is provided, zeroes are shifted out.
 * @rx_buf: Holds pointer to userspace buffer for receive data, or null.
 * @len: Length of tx and rx buffers, in bytes.
 * @speed_hz: Temporary override of the device's bitrate.
 * @bits_per_word: Temporary override of the device's wordsize.
 * @delay_usecs: If nonzero, how long to delay after the last bit transfer
 *	before optionally deselecting the device before the next transfer.
 * @cs_change: True to deselect device before starting the next transfer.
 *
 * This structure is mapped directly to the kernel spi_transfer structure;
 * the fields have the same meanings, except of course that the pointers
 * are in a different address space (and may be of different sizes in some
 * cases, such as 32-bit i386 userspace over a 64-bit x86_64 kernel).
 */
struct spi_ioc_transfer {
	__u64		tx_buf;
	__u64		rx_buf;

	__u32		len;
	__u32		speed_hz;

	__u16		delay_usecs;
	__u8		bits_per_word;
	__u8		cs_change;

	/* If the contents of 'struct spi_ioc_transfer' ever change
	 * incompatibly, then the ioctl number (currently 0) must change;
	 * ioctls with constant size fields get a bit more in the way of
	 * error checking than ones (like this) where that field varies.
	 *
	 * NOTE: struct layout is the same in 64bit and 32bit userspace.
	 */
};


/* Read of SPI mode (SPI_MODE_0..SPI_MODE_3) */
#define SPI_IOC_RD_MODE			_IOR(SPI_IOC_MAGIC, 1, __u8)

/* Read/Write Message */
#define SPI_IOC_WR_MSG			_IOW(SPI_IOC_MAGIC, 2, __u8)
#define SPI_IOC_RD_MSG			_IOR(SPI_IOC_MAGIC, 2, __u8)

/* Resend received Message */
#define SPI_IOC_RESEND			_IOR(SPI_IOC_MAGIC, 3, __u8)


//#define IRRC_DEBUG_MSG(fmt, args...)    printk(KERN_DEBUG "[IRRC] " fmt, ##args)
#define IRRC_DEBUG_MSG(fmt, args...) 
#define IRRC_INFO_MSG(fmt, args...)    printk(KERN_INFO "[IRRC] " fmt, ##args)
#define IRRC_ERR_MSG(fmt, args...) printk(KERN_ERR "[IRRC] [%s %d] " fmt, __FUNCTION__, __LINE__, ##args)

#endif
